# Multi Quiz System - Complete Guide

A professional online quiz/questionnaire system built with PHP and MySQL. Features admin management, student quiz-taking, automatic scoring, and detailed progress reports.

## Features

### 🎓 Admin Features
- **Dashboard** - Overview of statistics and quick access
- **Create Questionnaires** - Design multiple choice, true/false, and short-answer quizzes
- **Manage Questions** - Add, edit, and organize questions with flexible options
- **Assign Quizzes** - Distribute questionnaires to students
- **View Reports** - Comprehensive student and quiz performance analytics
- **Student Progress Tracking** - Monitor individual student performance

### 📚 Student Features
- **User Registration & Login** - Secure student account creation
- **Available Quizzes Dashboard** - View assigned questionnaires
- **Take Quiz** - Interactive quiz interface with timer
- **Auto-save** - Answers automatically saved during quiz
- **View Results** - Instant feedback with score breakdown
- **Progress Tracking** - Monitor performance history
- **Retake Quizzes** - Multiple attempts support

### 🎯 System Features
- **Responsive Design** - Works on desktop, tablet, and mobile
- **Automatic Scoring** - Instant grading for objective questions
- **Quiz Timer** - Countdown timer with auto-submit
- **Pass/Fail System** - Configurable passing percentages
- **User-Friendly UI** - Modern, intuitive interface
- **Security** - Password hashing, session management, input validation

## Installation

### Prerequisites
- PHP 7.4 or higher
- MySQL 5.7 or higher
- Apache/Nginx web server
- Composer (optional, for package management)

### Step 1: Set Up Database

1. Open phpMyAdmin or MySQL command line
2. Create a new database:
```sql
CREATE DATABASE quiz_system;
USE quiz_system;
```

3. Import the database schema by running this file: `database_schema.sql`
```bash
mysql -u root -p quiz_system < database_schema.sql
```

Or copy-paste the contents of `database_schema.sql` into phpMyAdmin's SQL tab.

### Step 2: Configure Database Connection

1. Edit `config.php` and update these lines with your database credentials:
```php
define('DB_HOST', 'localhost');    // Your database host
define('DB_USER', 'root');         // Your database username
define('DB_PASS', '');             // Your database password
define('DB_NAME', 'quiz_system');  // Your database name
```

### Step 3: Set Up Directory Structure

Create the following directories in your web root (e.g., `/var/www/html/`):

```
quiz_system/
├── admin/
├── auth/
├── classes/
├── includes/
├── public/
│   └── css/
├── student/
├── config.php
└── database_schema.sql
```

### Step 4: Move Files

1. Move all files to your web server directory
2. Ensure proper permissions (755 for directories, 644 for files)
3. Make sure `config.php` is readable by the web server

### Step 5: Access the Application

1. Open your browser and navigate to:
   - Main: `http://localhost/quiz_system/`
   - Will redirect to: `http://localhost/quiz_system/auth/login.php`

2. Create an admin account first (will need direct database insert or modify registration):
```sql
INSERT INTO users (username, email, password, full_name, role) 
VALUES ('admin', 'admin@example.com', '$2y$10$...', 'Administrator', 'admin');
```

Use PHP to generate the hashed password:
```php
echo password_hash('admin123', PASSWORD_BCRYPT);
```

3. Register as a student or have the admin create student accounts

## Usage

### For Admins

#### Creating a Questionnaire
1. Go to Questionnaires → Create New Questionnaire
2. Fill in title, description, duration, and passing percentage
3. Click "Create Questionnaire"
4. Add questions to the questionnaire
5. Set correct answers for each question
6. Publish the questionnaire
7. Assign to students

#### Adding Questions
1. Click "Questions" on a questionnaire
2. Select question type (Multiple Choice, True/False, Short Answer)
3. Enter question text and marks
4. Add options with correct answer marked
5. Save question

#### Assigning to Students
1. Click "Assign" on a questionnaire
2. Select students from the list
3. Click "Assign Selected Students"

#### Viewing Reports
1. Go to Reports
2. View overall statistics
3. Click on student name for detailed performance
4. View individual quiz results

### For Students

#### Taking a Quiz
1. Log in to your account
2. Go to Available Quizzes
3. Click "Start Quiz" on a questionnaire
4. Answer all questions (auto-saves)
5. Click "Submit Quiz" when complete
6. View your results immediately

#### Viewing Progress
1. Go to My Progress
2. See all quiz attempts and scores
3. Click on a quiz to review answers

## Database Schema

### Users Table
- `id` - Primary key
- `username` - Unique username
- `email` - User email
- `password` - Hashed password
- `full_name` - User's full name
- `role` - 'admin' or 'student'
- `created_at` - Registration timestamp

### Questionnaires Table
- `id` - Primary key
- `title` - Quiz title
- `description` - Quiz description
- `created_by` - Admin user ID
- `duration_minutes` - Quiz time limit
- `passing_percentage` - Required score to pass
- `is_published` - Publication status

### Questions Table
- `id` - Primary key
- `questionnaire_id` - Foreign key to questionnaires
- `question_text` - The question
- `question_type` - Type of question (multiple_choice, true_false, short_answer)
- `marks` - Points for this question
- `display_order` - Question order

### Options Table
- `id` - Primary key
- `question_id` - Foreign key to questions
- `option_text` - The option text
- `is_correct` - Whether this is the correct answer
- `option_order` - Option display order

### Student Questionnaire Table
- `id` - Primary key
- `student_id` - Student user ID
- `questionnaire_id` - Quiz ID
- `assigned_at` - Assignment date

### Student Attempts Table
- `id` - Primary key
- `student_id` - Student user ID
- `questionnaire_id` - Quiz ID
- `start_time` - Quiz start time
- `end_time` - Quiz submission time
- `obtained_marks` - Score received
- `total_marks` - Total possible marks
- `percentage` - Percentage score
- `status` - 'in_progress', 'submitted', 'graded'

### Student Answers Table
- `id` - Primary key
- `attempt_id` - Quiz attempt ID
- `question_id` - Question ID
- `selected_option_id` - Selected answer option
- `answer_text` - Free text answer
- `is_correct` - Whether answer is correct
- `marks_obtained` - Marks for this answer

## File Structure

```
config.php                           - Database configuration
database_schema.sql                  - Database creation script

classes/
  ├── Auth.php                       - Authentication & user management
  ├── Questionnaire.php              - Quiz management
  ├── Question.php                   - Question management
  ├── Option.php                     - Quiz options/answers
  ├── QuizAttempt.php               - Quiz attempt & scoring
  └── Report.php                     - Reporting & analytics

auth/
  ├── login.php                      - Login page
  ├── register.php                   - Registration page
  └── logout.php                     - Logout handler

admin/
  ├── dashboard.php                  - Admin dashboard
  ├── questionnaires.php             - Create/manage quizzes
  ├── questions.php                  - Manage questions
  ├── options.php                    - Manage quiz options
  ├── assign-students.php            - Assign quizzes to students
  ├── reports.php                    - View analytics
  ├── student-detail.php             - Individual student report
  ├── quiz-detail.php                - Individual quiz report
  ├── edit-questionnaire.php         - Edit quiz details
  ├── edit-question.php              - Edit questions
  └── publish-questionnaire.php      - Publish quiz

student/
  ├── dashboard.php                  - Student home
  ├── quizzes.php                    - Available quizzes list
  ├── take-quiz.php                  - Quiz interface
  ├── quiz-result.php                - Results & review
  └── progress.php                   - Progress tracking

includes/
  └── header.php                     - Navigation header

public/css/
  └── style.css                      - Main stylesheet
```

## Security Features

- **Password Hashing** - Bcrypt hashing for secure passwords
- **Session Management** - Secure PHP sessions
- **Input Validation** - All inputs validated and sanitized
- **SQL Injection Prevention** - Prepared statements used throughout
- **CSRF Protection Ready** - Can be added with token implementation
- **Role-Based Access** - Admin and student role verification

## Troubleshooting

### Issue: "Connection failed" error
**Solution:** Check database credentials in `config.php` match your MySQL setup

### Issue: Blank pages
**Solution:** 
- Check PHP error logs: `php -r "php_uname();"`
- Enable error reporting in `config.php`
- Ensure all class files are in correct directories

### Issue: Files not uploading
**Solution:** Check directory permissions, ensure write access to directories

### Issue: Redirects not working
**Solution:** 
- Check `BASE_URL` in `config.php` matches your installation
- Ensure `header()` calls are before any output

### Issue: Timer not working
**Solution:** Check browser console for JavaScript errors, ensure JavaScript is enabled

## Performance Optimization

1. **Add Indexes** - Key database fields already indexed
2. **Caching** - Consider Redis for session storage
3. **Database Optimization** - Run `OPTIMIZE TABLE` periodically
4. **Compression** - Enable gzip in Apache/Nginx

## Backup & Restore

### Backup Database
```bash
mysqldump -u root -p quiz_system > quiz_system_backup.sql
```

### Restore Database
```bash
mysql -u root -p quiz_system < quiz_system_backup.sql
```

## Future Enhancements

- [ ] Question bank with question pooling
- [ ] Randomized question order
- [ ] Negative marking
- [ ] Bulk import/export
- [ ] Email notifications
- [ ] Certificate generation
- [ ] Analytics dashboard
- [ ] Mobile app
- [ ] API endpoints
- [ ] Two-factor authentication

## License

This project is provided as-is for educational and commercial use.

## Support

For issues or questions:
1. Check the troubleshooting section
2. Review error logs
3. Verify database connectivity
4. Ensure all files are in correct locations

## Credits

Built with PHP 7.4+, MySQL, and HTML5/CSS3
