<?php
class Auth {
    private $conn;
    private $table = 'users';
    
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Register new user
    public function register($username, $email, $full_name, $password, $role = 'student') {
        // Check if user exists
        $query = "SELECT id FROM " . $this->table . " WHERE username = ? OR email = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ss", $username, $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return array('success' => false, 'message' => 'Username or email already exists');
        }
        
        // Hash password
        $hashed_password = password_hash($password, PASSWORD_BCRYPT);
        
        // Insert user
        $query = "INSERT INTO " . $this->table . " (username, email, full_name, password, role) 
                  VALUES (?, ?, ?, ?, ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("sssss", $username, $email, $full_name, $hashed_password, $role);
        
        if ($stmt->execute()) {
            return array('success' => true, 'message' => 'Registration successful');
        } else {
            return array('success' => false, 'message' => 'Registration failed');
        }
    }
    
    // Login user
    public function login($username, $password) {
        $query = "SELECT id, username, email, full_name, role FROM " . $this->table . " 
                  WHERE username = ? OR email = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ss", $username, $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $row = $result->fetch_assoc();
            
            // Get password from database
            $query = "SELECT password FROM " . $this->table . " WHERE id = ?";
            $stmt = $this->conn->prepare($query);
            $stmt->bind_param("i", $row['id']);
            $stmt->execute();
            $pwd_result = $stmt->get_result();
            $pwd_row = $pwd_result->fetch_assoc();
            
            if (password_verify($password, $pwd_row['password'])) {
                $_SESSION['user_id'] = $row['id'];
                $_SESSION['username'] = $row['username'];
                $_SESSION['email'] = $row['email'];
                $_SESSION['full_name'] = $row['full_name'];
                $_SESSION['role'] = $row['role'];
                
                return array('success' => true, 'message' => 'Login successful', 'role' => $row['role']);
            }
        }
        
        return array('success' => false, 'message' => 'Invalid credentials');
    }
    
    // Logout user
    public function logout() {
        session_destroy();
        return array('success' => true, 'message' => 'Logged out successfully');
    }
    
    // Get user by ID
    public function getUserById($id) {
        $query = "SELECT * FROM " . $this->table . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
    
    // Update user profile
    public function updateProfile($id, $full_name, $email) {
        $query = "UPDATE " . $this->table . " SET full_name = ?, email = ? WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ssi", $full_name, $email, $id);
        
        if ($stmt->execute()) {
            $_SESSION['full_name'] = $full_name;
            $_SESSION['email'] = $email;
            return array('success' => true, 'message' => 'Profile updated successfully');
        }
        return array('success' => false, 'message' => 'Update failed');
    }
}
?>
