<?php
class Questionnaire {
    private $conn;
    private $table = 'questionnaires';
    
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Create new questionnaire
    public function create($title, $description, $created_by, $duration_minutes, $passing_percentage) {
        $query = "INSERT INTO " . $this->table . " 
                  (title, description, created_by, duration_minutes, passing_percentage) 
                  VALUES (?, ?, ?, ?, ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ssiii", $title, $description, $created_by, $duration_minutes, $passing_percentage);
        
        if ($stmt->execute()) {
            return array('success' => true, 'id' => $this->conn->insert_id);
        }
        return array('success' => false);
    }
    
    // Get all questionnaires for admin
    public function getAllQuestionnaires($admin_id = null) {
        $query = "SELECT q.*, COUNT(q.id) as total_questions, u.full_name as created_by_name
                  FROM " . $this->table . " q
                  LEFT JOIN users u ON q.created_by = u.id";
        
        if ($admin_id) {
            $query .= " WHERE q.created_by = ?";
        }
        
        $query .= " GROUP BY q.id ORDER BY q.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        if ($admin_id) {
            $stmt->bind_param("i", $admin_id);
        }
        $stmt->execute();
        return $stmt->get_result();
    }
    
    // Get questionnaire by ID
    public function getById($id) {
        $query = "SELECT * FROM " . $this->table . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
    
    // Update questionnaire
    public function update($id, $title, $description, $duration_minutes, $passing_percentage) {
        $query = "UPDATE " . $this->table . " 
                  SET title = ?, description = ?, duration_minutes = ?, passing_percentage = ?
                  WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ssiii", $title, $description, $duration_minutes, $passing_percentage, $id);
        
        if ($stmt->execute()) {
            return array('success' => true);
        }
        return array('success' => false);
    }
    
    // Publish questionnaire
    public function publish($id) {
        $query = "UPDATE " . $this->table . " SET is_published = 1 WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            return array('success' => true);
        }
        return array('success' => false);
    }
    
    // Delete questionnaire
    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            return array('success' => true);
        }
        return array('success' => false);
    }
    
    // Get questionnaire with questions
    public function getWithQuestions($id) {
        $questionnaire = $this->getById($id);
        
        if ($questionnaire) {
            $query = "SELECT q.*, COUNT(o.id) as total_options
                      FROM questions q
                      LEFT JOIN options o ON q.id = o.question_id
                      WHERE q.questionnaire_id = ?
                      GROUP BY q.id
                      ORDER BY q.display_order ASC";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $questionnaire['questions'] = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        }
        
        return $questionnaire;
    }
    
    // Assign questionnaire to students
    public function assignToStudents($questionnaire_id, $student_ids) {
        $query = "INSERT INTO student_questionnaire (student_id, questionnaire_id) VALUES (?, ?)";
        $stmt = $this->conn->prepare($query);
        
        $success_count = 0;
        foreach ($student_ids as $student_id) {
            $stmt->bind_param("ii", $student_id, $questionnaire_id);
            if ($stmt->execute()) {
                $success_count++;
            }
        }
        
        return array('success' => true, 'assigned' => $success_count);
    }
    
    // Get assigned students
    public function getAssignedStudents($questionnaire_id) {
        $query = "SELECT u.id, u.full_name, u.email, sq.assigned_at,
                  COUNT(sa.id) as attempts,
                  (SELECT obtained_marks FROM student_attempts 
                   WHERE student_id = u.id AND questionnaire_id = ? 
                   ORDER BY end_time DESC LIMIT 1) as last_score
                  FROM users u
                  JOIN student_questionnaire sq ON u.id = sq.student_id
                  LEFT JOIN student_attempts sa ON u.id = sa.student_id AND sa.questionnaire_id = ?
                  WHERE sq.questionnaire_id = ?
                  GROUP BY u.id
                  ORDER BY u.full_name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("iii", $questionnaire_id, $questionnaire_id, $questionnaire_id);
        $stmt->execute();
        return $stmt->get_result();
    }
}
?>
