<?php
class QuizAttempt {
    private $conn;
    private $attempt_table = 'student_attempts';
    private $answer_table = 'student_answers';
    
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Start a new quiz attempt
    public function startAttempt($student_id, $questionnaire_id) {
        $start_time = date('Y-m-d H:i:s');
        
        $query = "INSERT INTO " . $this->attempt_table . " 
                  (student_id, questionnaire_id, start_time, status) 
                  VALUES (?, ?, ?, 'in_progress')";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("iis", $student_id, $questionnaire_id, $start_time);
        
        if ($stmt->execute()) {
            return array('success' => true, 'attempt_id' => $this->conn->insert_id);
        }
        return array('success' => false);
    }
    
    // Get current attempt
    public function getCurrentAttempt($student_id, $questionnaire_id) {
        $query = "SELECT * FROM " . $this->attempt_table . " 
                  WHERE student_id = ? AND questionnaire_id = ? AND status = 'in_progress'
                  ORDER BY created_at DESC LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ii", $student_id, $questionnaire_id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
    
    // Save answer
    public function saveAnswer($attempt_id, $question_id, $selected_option_id = null, $answer_text = null) {
        $query = "INSERT INTO " . $this->answer_table . " 
                  (attempt_id, question_id, selected_option_id, answer_text) 
                  VALUES (?, ?, ?, ?)
                  ON DUPLICATE KEY UPDATE 
                  selected_option_id = ?, answer_text = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("iiisss", $attempt_id, $question_id, $selected_option_id, $answer_text, $selected_option_id, $answer_text);
        
        if ($stmt->execute()) {
            return array('success' => true);
        }
        return array('success' => false);
    }
    
    // Submit quiz
    public function submitQuiz($attempt_id) {
        $end_time = date('Y-m-d H:i:s');
        
        $query = "UPDATE " . $this->attempt_table . " 
                  SET end_time = ?, status = 'submitted'
                  WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("si", $end_time, $attempt_id);
        
        if ($stmt->execute()) {
            return $this->gradeQuiz($attempt_id);
        }
        return array('success' => false);
    }
    
    // Grade the quiz
    private function gradeQuiz($attempt_id) {
        // Get all answers for this attempt
        $query = "SELECT sa.*, q.marks, o.is_correct
                  FROM " . $this->answer_table . " sa
                  JOIN questions q ON sa.question_id = q.id
                  LEFT JOIN options o ON sa.selected_option_id = o.id
                  WHERE sa.attempt_id = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $attempt_id);
        $stmt->execute();
        $answers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        
        $total_marks = 0;
        $obtained_marks = 0;
        
        foreach ($answers as $answer) {
            $total_marks += $answer['marks'];
            
            if ($answer['is_correct']) {
                $obtained_marks += $answer['marks'];
                $is_correct = 1;
            } else {
                $is_correct = 0;
            }
            
            // Update answer with marks
            $query = "UPDATE " . $this->answer_table . " 
                      SET is_correct = ?, marks_obtained = ? 
                      WHERE id = ?";
            $stmt = $this->conn->prepare($query);
            $stmt->bind_param("iii", $is_correct, $answer['marks'], $answer['id']);
            $stmt->execute();
        }
        
        // Calculate percentage
        $percentage = ($total_marks > 0) ? ($obtained_marks / $total_marks) * 100 : 0;
        
        // Update attempt with scores
        $query = "UPDATE " . $this->attempt_table . " 
                  SET total_marks = ?, obtained_marks = ?, percentage = ?, status = 'graded'
                  WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("dddi", $total_marks, $obtained_marks, $percentage, $attempt_id);
        
        if ($stmt->execute()) {
            return array('success' => true, 'obtained_marks' => $obtained_marks, 'total_marks' => $total_marks, 'percentage' => $percentage);
        }
        return array('success' => false);
    }
    
    // Get student attempts
    public function getStudentAttempts($student_id, $questionnaire_id = null) {
        $query = "SELECT sa.*, q.title as questionnaire_title 
                  FROM " . $this->attempt_table . " sa
                  JOIN questionnaires q ON sa.questionnaire_id = q.id
                  WHERE sa.student_id = ?";
        
        if ($questionnaire_id) {
            $query .= " AND sa.questionnaire_id = ?";
        }
        
        $query .= " ORDER BY sa.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        
        if ($questionnaire_id) {
            $stmt->bind_param("ii", $student_id, $questionnaire_id);
        } else {
            $stmt->bind_param("i", $student_id);
        }
        
        $stmt->execute();
        return $stmt->get_result();
    }
    
    // Get attempt details
    public function getAttemptDetails($attempt_id) {
        $query = "SELECT * FROM " . $this->attempt_table . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $attempt_id);
        $stmt->execute();
        $attempt = $stmt->get_result()->fetch_assoc();
        
        if ($attempt) {
            // Get all answers with question details
            $query = "SELECT sa.*, q.question_text, q.marks, o.option_text, opt.is_correct
                      FROM " . $this->answer_table . " sa
                      JOIN questions q ON sa.question_id = q.id
                      LEFT JOIN options o ON sa.selected_option_id = o.id
                      LEFT JOIN options opt ON sa.selected_option_id = opt.id
                      WHERE sa.attempt_id = ?
                      ORDER BY q.display_order";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bind_param("i", $attempt_id);
            $stmt->execute();
            $attempt['answers'] = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        }
        
        return $attempt;
    }
    
    // Get attempt by ID
    public function getById($id) {
        $query = "SELECT * FROM " . $this->attempt_table . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
}
?>
