<?php
class Report {
    private $conn;
    
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Get all students report
    public function getAllStudentsReport() {
        $query = "SELECT u.id, u.full_name, u.email, u.created_at,
                  COUNT(DISTINCT sa.questionnaire_id) as quizzes_attempted,
                  COUNT(sa.id) as total_attempts,
                  AVG(sa.percentage) as average_percentage,
                  MAX(sa.percentage) as highest_score,
                  MIN(sa.percentage) as lowest_score
                  FROM users u
                  LEFT JOIN student_attempts sa ON u.id = sa.student_id AND sa.status = 'graded'
                  WHERE u.role = 'student'
                  GROUP BY u.id
                  ORDER BY u.full_name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->get_result();
    }
    
    // Get individual student report
    public function getStudentReport($student_id) {
        $query = "SELECT u.id, u.full_name, u.email, u.created_at,
                  COUNT(DISTINCT sa.questionnaire_id) as quizzes_attempted,
                  COUNT(sa.id) as total_attempts,
                  AVG(sa.percentage) as average_percentage,
                  MAX(sa.percentage) as highest_score,
                  MIN(sa.percentage) as lowest_score
                  FROM users u
                  LEFT JOIN student_attempts sa ON u.id = sa.student_id AND sa.status = 'graded'
                  WHERE u.id = ? AND u.role = 'student'
                  GROUP BY u.id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
    
    // Get student quiz attempts
    public function getStudentQuizAttempts($student_id) {
        $query = "SELECT sa.id, sa.questionnaire_id, q.title as questionnaire_title,
                  sa.start_time, sa.end_time, sa.obtained_marks, sa.total_marks,
                  sa.percentage, sa.status
                  FROM student_attempts sa
                  JOIN questionnaires q ON sa.questionnaire_id = q.id
                  WHERE sa.student_id = ? AND sa.status = 'graded'
                  ORDER BY sa.end_time DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        return $stmt->get_result();
    }
    
    // Get questionnaire report
    public function getQuestionnaireReport($questionnaire_id) {
        $query = "SELECT q.id, q.title, q.total_questions, q.passing_percentage,
                  COUNT(DISTINCT sa.student_id) as students_attempted,
                  COUNT(sa.id) as total_attempts,
                  AVG(sa.percentage) as average_percentage,
                  MAX(sa.percentage) as highest_score,
                  MIN(sa.percentage) as lowest_score,
                  SUM(CASE WHEN sa.percentage >= q.passing_percentage THEN 1 ELSE 0 END) as passed_count
                  FROM questionnaires q
                  LEFT JOIN student_attempts sa ON q.id = sa.questionnaire_id AND sa.status = 'graded'
                  WHERE q.id = ?
                  GROUP BY q.id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $questionnaire_id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
    
    // Get questionnaire student results
    public function getQuestionnaireStudentResults($questionnaire_id) {
        $query = "SELECT u.id, u.full_name, u.email,
                  COUNT(sa.id) as attempts,
                  MAX(sa.percentage) as best_score,
                  MAX(sa.obtained_marks) as best_marks,
                  MAX(sa.total_marks) as total_marks,
                  MAX(CASE WHEN sa.percentage >= q.passing_percentage THEN 'Pass' ELSE 'Fail' END) as status,
                  MAX(sa.end_time) as last_attempt
                  FROM users u
                  JOIN student_questionnaire sq ON u.id = sq.student_id
                  LEFT JOIN student_attempts sa ON u.id = sa.student_id AND sa.questionnaire_id = sq.questionnaire_id AND sa.status = 'graded'
                  JOIN questionnaires q ON sq.questionnaire_id = q.id
                  WHERE sq.questionnaire_id = ?
                  GROUP BY u.id
                  ORDER BY best_score DESC, u.full_name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $questionnaire_id);
        $stmt->execute();
        return $stmt->get_result();
    }
    
    // Get class statistics
    public function getClassStatistics() {
        $query = "SELECT 
                  COUNT(DISTINCT u.id) as total_students,
                  COUNT(DISTINCT q.id) as total_questionnaires,
                  COUNT(DISTINCT sa.id) as total_attempts,
                  AVG(sa.percentage) as class_average,
                  MAX(sa.percentage) as highest_score,
                  MIN(sa.percentage) as lowest_score
                  FROM users u
                  CROSS JOIN questionnaires q
                  LEFT JOIN student_attempts sa ON u.id = sa.student_id AND sa.status = 'graded'
                  WHERE u.role = 'student'";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
    
    // Export student progress to CSV
    public function exportStudentProgress($student_id) {
        $query = "SELECT sa.id, q.title, sa.start_time, sa.end_time,
                  sa.obtained_marks, sa.total_marks, sa.percentage, sa.status
                  FROM student_attempts sa
                  JOIN questionnaires q ON sa.questionnaire_id = q.id
                  WHERE sa.student_id = ?
                  ORDER BY sa.end_time DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        return $stmt->get_result();
    }
}
?>
